<?php

namespace Tests\Feature\Admin;

use App\Models\Category;
use App\Models\Product;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class AdminCategoryControllerTest extends TestCase
{
    use RefreshDatabase;

    private User $admin;

    protected function setUp(): void
    {
        parent::setUp();

        $this->admin = User::factory()->create([
            'is_admin' => true,
            'admin_approved_at' => now(),
        ]);

        $this->actingAs($this->admin);
    }

    public function test_admin_can_view_categories_list(): void
    {
        Category::factory()->count(5)->create();

        $response = $this->get(route('admin.categories.index'));

        $response->assertStatus(200);
        $response->assertViewIs('admin.categories.index');
        $response->assertViewHas('categories');
    }

    public function test_admin_can_view_category_creation_form(): void
    {
        $response = $this->get(route('admin.categories.create'));

        $response->assertStatus(200);
        $response->assertViewIs('admin.categories.create');
        $response->assertViewHas('parentCategories');
    }

    public function test_admin_can_create_category(): void
    {
        $categoryData = [
            'name_ar' => 'إلكترونيات',
            'name_en' => 'Electronics',
            'slug' => 'electronics',
            'status' => 'active',
        ];

        $response = $this->post(route('admin.categories.store'), $categoryData);

        $response->assertRedirect(route('admin.categories.index'));
        $response->assertSessionHas('success');
        $this->assertDatabaseHas('categories', [
            'name_en' => 'Electronics',
            'slug' => 'electronics',
        ]);
    }

    public function test_admin_can_create_category_with_parent(): void
    {
        $parent = Category::factory()->create();

        $categoryData = [
            'parent_id' => $parent->id,
            'name_ar' => 'هواتف',
            'name_en' => 'Phones',
            'slug' => 'phones',
            'status' => 'active',
        ];

        $response = $this->post(route('admin.categories.store'), $categoryData);

        $response->assertRedirect(route('admin.categories.index'));
        $this->assertDatabaseHas('categories', [
            'parent_id' => $parent->id,
            'name_en' => 'Phones',
        ]);
    }

    public function test_admin_can_edit_category(): void
    {
        $category = Category::factory()->create();

        $response = $this->get(route('admin.categories.edit', $category));

        $response->assertStatus(200);
        $response->assertViewIs('admin.categories.edit');
        $response->assertViewHas('category');
        $response->assertViewHas('parentCategories');
    }

    public function test_admin_can_update_category(): void
    {
        $category = Category::factory()->create();

        $updatedData = [
            'name_ar' => 'إلكترونيات محدثة',
            'name_en' => 'Updated Electronics',
            'slug' => 'updated-electronics',
            'status' => 'active',
        ];

        $response = $this->put(route('admin.categories.update', $category), $updatedData);

        $response->assertRedirect(route('admin.categories.index'));
        $response->assertSessionHas('success');
        $this->assertDatabaseHas('categories', [
            'id' => $category->id,
            'name_en' => 'Updated Electronics',
        ]);
    }

    public function test_admin_cannot_delete_category_with_products(): void
    {
        $category = Category::factory()->create();
        Product::factory()->create(['category_id' => $category->id]);

        $response = $this->delete(route('admin.categories.destroy', $category));

        $response->assertRedirect(route('admin.categories.index'));
        $response->assertSessionHas('error');
        $this->assertDatabaseHas('categories', [
            'id' => $category->id,
        ]);
    }

    public function test_admin_cannot_delete_category_with_children(): void
    {
        $parent = Category::factory()->create();
        Category::factory()->create(['parent_id' => $parent->id]);

        $response = $this->delete(route('admin.categories.destroy', $parent));

        $response->assertRedirect(route('admin.categories.index'));
        $response->assertSessionHas('error');
        $this->assertDatabaseHas('categories', [
            'id' => $parent->id,
        ]);
    }

    public function test_admin_can_delete_category_without_products_or_children(): void
    {
        $category = Category::factory()->create();

        $response = $this->delete(route('admin.categories.destroy', $category));

        $response->assertRedirect(route('admin.categories.index'));
        $response->assertSessionHas('success');
        $this->assertDatabaseMissing('categories', [
            'id' => $category->id,
        ]);
    }

    public function test_admin_cannot_create_category_without_required_fields(): void
    {
        $response = $this->post(route('admin.categories.store'), []);

        $response->assertSessionHasErrors(['name_ar', 'name_en', 'slug', 'status']);
    }
}
