<?php

namespace Tests\Feature\Admin;

use App\Models\Product;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class AdminDashboardTest extends TestCase
{
    use RefreshDatabase;

    private User $admin;

    protected function setUp(): void
    {
        parent::setUp();

        $this->admin = User::factory()->create([
            'is_admin' => true,
            'admin_approved_at' => now(),
        ]);

        $this->actingAs($this->admin);
    }

    public function test_admin_can_view_dashboard(): void
    {
        $response = $this->get(route('admin.dashboard'));

        $response->assertStatus(200);
        $response->assertViewIs('admin.dashboard.index');
    }

    public function test_dashboard_displays_correct_statistics(): void
    {
        Product::factory()->count(15)->create();

        $response = $this->get(route('admin.dashboard'));

        $response->assertStatus(200);
        $stats = $response->viewData('stats');

        $this->assertEquals(15, $stats['overview']['total_products']);
        $this->assertGreaterThan(0, $stats['overview']['total_categories']);
    }

    public function test_dashboard_displays_products_by_source(): void
    {
        Product::factory()->count(5)->create(['source' => 'aliexpress']);
        Product::factory()->count(3)->create(['source' => 'amazon']);
        Product::factory()->count(2)->create(['source' => 'noon']);

        $response = $this->get(route('admin.dashboard'));

        $response->assertStatus(200);
        $stats = $response->viewData('stats');

        $this->assertArrayHasKey('products_by_source', $stats);
        $this->assertCount(3, $stats['overview']['products_by_source']);
    }

    public function test_dashboard_displays_recent_activity(): void
    {
        Product::factory()->count(10)->create();

        $response = $this->get(route('admin.dashboard'));

        $response->assertStatus(200);
        $stats = $response->viewData('stats');

        $this->assertCount(10, $stats['recent_activity']);
        $this->assertInstanceOf(Product::class, $stats['recent_activity']->first());
    }

    public function test_dashboard_displays_price_distribution(): void
    {
        Product::factory()->count(5)->create(['price' => 50]);
        Product::factory()->count(3)->create(['price' => 250]);
        Product::factory()->count(2)->create(['price' => 750]);
        Product::factory()->count(1)->create(['price' => 1500]);

        $response = $this->get(route('admin.dashboard'));

        $response->assertStatus(200);
        $stats = $response->viewData('stats');

        $this->assertArrayHasKey('price_distribution', $stats);
        $this->assertArrayHasKey('labels', $stats['price_distribution']);
        $this->assertArrayHasKey('datasets', $stats['price_distribution']);
    }

    public function test_dashboard_displays_products_over_time(): void
    {
        Product::factory()->count(20)->create();

        $response = $this->get(route('admin.dashboard'));

        $response->assertStatus(200);
        $stats = $response->viewData('stats');

        $this->assertArrayHasKey('products_over_time', $stats);
        $this->assertArrayHasKey('labels', $stats['products_over_time']);
        $this->assertArrayHasKey('datasets', $stats['products_over_time']);
    }
}
