<?php

declare(strict_types=1);

namespace Tests\Feature;

use App\Models\User;
use App\Services\AiAssistant\AiAssistantService;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class AiAssistantServiceTest extends TestCase
{
    use RefreshDatabase;

    private ?AiAssistantService $assistant = null;

    private ?User $user = null;

    protected function setUp(): void
    {
        parent::setUp();

        $this->assistant = app(AiAssistantService::class);
        $this->user = User::factory()->create();
    }

    public function test_chat_creates_new_conversation(): void
    {
        // Skip if API key not configured
        if (! $this->hasApiKey()) {
            $this->markTestSkipped('AI API key not configured');
        }

        $result = $this->assistant->chat(
            message: 'Hello, what can you help me with?',
            user: $this->user
        );

        $this->assertArrayHasKey('content', $result);
        $this->assertArrayHasKey('conversation_id', $result);
        $this->assertNotEmpty($result['conversation_id']);

        // Verify conversation was created
        $this->assertDatabaseHas('ai_conversations', [
            'user_id' => $this->user->id,
        ]);
    }

    public function test_chat_continues_existing_conversation(): void
    {
        if (! $this->hasApiKey()) {
            $this->markTestSkipped('AI API key not configured');
        }

        $result1 = $this->assistant->chat(
            message: 'What products do you have?',
            user: $this->user
        );

        $conversationId = $result1['conversation_id'];

        $result2 = $this->assistant->chat(
            message: 'Tell me more about deals',
            conversationUlid: $conversationId,
            user: $this->user
        );

        $this->assertEquals($conversationId, $result2['conversation_id']);
    }

    public function test_get_conversations(): void
    {
        // Create a conversation
        $this->assistant->chat(
            message: 'Test message',
            user: $this->user
        );

        $conversations = $this->assistant->getConversations($this->user);

        $this->assertIsArray($conversations);
        $this->assertCount(1, $conversations);
        $this->assertArrayHasKey('id', $conversations[0]);
        $this->assertArrayHasKey('title', $conversations[0]);
    }

    public function test_delete_conversation(): void
    {
        $result = $this->assistant->chat(
            message: 'Test message',
            user: $this->user
        );

        $conversationId = $result['conversation_id'];

        $deleted = $this->assistant->deleteConversation($conversationId);

        $this->assertTrue($deleted);

        $this->assertDatabaseMissing('ai_conversations', [
            'ulid' => $conversationId,
        ]);
    }

    protected function hasApiKey(): bool
    {
        return ! empty(config('services.deepseek.api_key'))
            || ! empty(config('services.mistral.api_key'));
    }
}
