<?php

namespace Tests\Feature\Auth;

use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class RegistrationTest extends TestCase
{
    use RefreshDatabase;

    public function test_user_can_register_with_email(): void
    {
        $response = $this->postJson('/api/v1/register', [
            'first_name' => 'John',
            'last_name' => 'Doe',
            'email' => 'john@example.com',
            'password' => 'password123',
            'password_confirmation' => 'password123',
        ]);

        $response->assertCreated();
        $response->assertJsonStructure([
            'user' => ['id', 'first_name', 'last_name', 'email', 'phone'],
            'access_token',
            'token_type',
        ]);

        $this->assertDatabaseHas('users', [
            'email' => 'john@example.com',
            'first_name' => 'John',
            'last_name' => 'Doe',
        ]);
    }

    public function test_user_can_register_with_phone(): void
    {
        $response = $this->postJson('/api/v1/register', [
            'first_name' => 'Jane',
            'last_name' => 'Smith',
            'phone' => '+1234567890',
            'password' => 'password123',
            'password_confirmation' => 'password123',
        ]);

        $response->assertCreated();
        $response->assertJsonStructure([
            'user' => ['id', 'first_name', 'last_name', 'email', 'phone'],
            'access_token',
            'token_type',
        ]);

        $this->assertDatabaseHas('users', [
            'phone' => '+1234567890',
            'first_name' => 'Jane',
            'last_name' => 'Smith',
        ]);
    }

    public function test_user_can_register_with_both_email_and_phone(): void
    {
        $response = $this->postJson('/api/v1/register', [
            'first_name' => 'Bob',
            'last_name' => 'Johnson',
            'email' => 'bob@example.com',
            'phone' => '+9876543210',
            'password' => 'password123',
            'password_confirmation' => 'password123',
        ]);

        $response->assertCreated();
        $this->assertDatabaseHas('users', [
            'email' => 'bob@example.com',
            'phone' => '+9876543210',
        ]);
    }

    public function test_registration_requires_email_or_phone(): void
    {
        $response = $this->postJson('/api/v1/register', [
            'first_name' => 'Alice',
            'last_name' => 'Brown',
            'password' => 'password123',
            'password_confirmation' => 'password123',
        ]);

        $response->assertUnprocessable();
        $response->assertJsonValidationErrors(['email', 'phone']);
    }

    public function test_registration_validates_unique_email(): void
    {
        User::factory()->create(['email' => 'existing@example.com']);

        $response = $this->postJson('/api/v1/register', [
            'first_name' => 'Test',
            'last_name' => 'User',
            'email' => 'existing@example.com',
            'password' => 'password123',
            'password_confirmation' => 'password123',
        ]);

        $response->assertUnprocessable();
        $response->assertJsonValidationErrors('email');
    }

    public function test_registration_validates_unique_phone(): void
    {
        User::factory()->create(['phone' => '+1111111111']);

        $response = $this->postJson('/api/v1/register', [
            'first_name' => 'Test',
            'last_name' => 'User',
            'phone' => '+1111111111',
            'password' => 'password123',
            'password_confirmation' => 'password123',
        ]);

        $response->assertUnprocessable();
        $response->assertJsonValidationErrors('phone');
    }

    public function test_registration_requires_password_confirmation(): void
    {
        $response = $this->postJson('/api/v1/register', [
            'first_name' => 'Test',
            'last_name' => 'User',
            'email' => 'test@example.com',
            'password' => 'password123',
            'password_confirmation' => 'wrongpassword',
        ]);

        $response->assertUnprocessable();
        $response->assertJsonValidationErrors('password');
    }

    public function test_registration_issues_access_token(): void
    {
        $response = $this->postJson('/api/v1/register', [
            'first_name' => 'Token',
            'last_name' => 'User',
            'email' => 'token@example.com',
            'password' => 'password123',
            'password_confirmation' => 'password123',
        ]);

        $response->assertCreated();
        $this->assertNotEmpty($response->json('access_token'));
        $this->assertEquals('Bearer', $response->json('token_type'));
    }
}
