<?php

namespace App\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;
use Illuminate\Support\Facades\Log;

class SendPasswordResetNotification extends Notification implements ShouldQueue
{
    use Queueable;

    /**
     * Create a new notification instance.
     */
    public function __construct(public string $token) {}

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        $channels = [];

        // Always use mail if email exists
        if ($notifiable->email) {
            $channels[] = 'mail';
        }

        // TODO: Add SMS channel for phone-only users
        // For now, phone users will receive reset codes via log
        // To enable SMS, install an SMS notification channel package
        // and configure it in config/services.php
        // Example: if ($notifiable->phone && !$notifiable->email) { $channels[] = 'vonage'; }

        return $channels;
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        return (new MailMessage)
            ->subject('Reset Password Notification')
            ->greeting('Hello '.$notifiable->first_name.',')
            ->line('You are receiving this email because we received a password reset request for your account.')
            ->action('Reset Password', url('/api/v1/reset-password/'.$this->token.'?email='.$notifiable->email))
            ->line('This password reset link will expire in 60 minutes.')
            ->line('If you did not request a password reset, no further action is required.')
            ->salutation('Regards, '.config('app.name'));
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            'token' => $this->token,
        ];
    }

    /**
     * Handle notification for phone-only users (log the code).
     * This is a temporary solution until SMS is configured.
     */
    public function toLog(object $notifiable): string
    {
        $message = sprintf(
            'Password reset code for %s: %s (expires in 60 minutes)',
            $notifiable->phone ?? $notifiable->email,
            $this->token
        );

        Log::info($message);

        return $message;
    }
}
