<?php

declare(strict_types=1);

namespace App\Services\AiAssistant;

use App\Models\Product;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Facades\View;

class AiComponentRenderer
{
    protected string $locale = 'en';

    /**
     * Set the locale for rendering components.
     */
    public function setLocale(string $locale): self
    {
        $this->locale = in_array($locale, ['ar', 'en']) ? $locale : 'en';

        return $this;
    }

    /**
     * Detect locale from a query string.
     */
    public function detectLocale(string $query): string
    {
        // Check if the query contains Arabic characters
        if (preg_match('/[\x{0600}-\x{06FF}]/u', $query)) {
            return 'ar';
        }

        return 'en';
    }

    /**
     * Auto-detect locale from query and set it.
     */
    public function detectAndSetLocale(string $query): self
    {
        return $this->setLocale($this->detectLocale($query));
    }

    /**
     * Get the appropriate name field based on current locale.
     */
    public function getName(Product $product): string
    {
        return $this->locale === 'ar'
            ? ($product->name_ar ?? $product->name_en)
            : $product->name_en;
    }

    /**
     * Get the appropriate description field based on current locale.
     */
    public function getDescription(Product $product): ?string
    {
        return $this->locale === 'ar'
            ? ($product->description_ar ?? $product->description_en)
            : $product->description_en;
    }

    public function productCard(Product $product, ?string $badge = null): string
    {
        return View::make('components.ai.product-card', [
            'product' => $product,
            'badge' => $badge,
            'locale' => $this->locale,
        ])->render();
    }

    public function productGrid(Collection $products, ?string $title = null, ?string $badge = null): string
    {
        return View::make('components.ai.product-grid', [
            'products' => $products,
            'title' => $title,
            'badge' => $badge,
            'locale' => $this->locale,
        ])->render();
    }

    public function comparisonTable(Collection $products): string
    {
        return View::make('components.ai.comparison-table', [
            'products' => $products,
            'locale' => $this->locale,
        ])->render();
    }

    public function dealCard(Product $product, string $dealType): string
    {
        return View::make('components.ai.deal-card', [
            'product' => $product,
            'dealType' => $dealType,
            'locale' => $this->locale,
        ])->render();
    }

    public function categoryGrid(Collection $categories): string
    {
        return View::make('components.ai.category-grid', [
            'categories' => $categories,
            'locale' => $this->locale,
        ])->render();
    }

    public function trendingBadge(): string
    {
        return View::make('components.ai.trending-badge')->render();
    }

    public function discountBadge(Product $product): string
    {
        if (! $product->discount || $product->discount <= 0) {
            return '';
        }

        return View::make('components.ai.discount-badge', [
            'discount' => $product->discount,
        ])->render();
    }
}
