<?php

namespace Tests\Feature\Auth;

use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class ProfileTest extends TestCase
{
    use RefreshDatabase;

    public function test_authenticated_user_can_create_profile(): void
    {
        $user = User::factory()->create();
        $token = $user->createToken('auth-token')->plainTextToken;

        $response = $this->postJson('/api/v1/profile', [
            'gender' => 'male',
            'birth_date' => '1990-01-15',
            'city' => 'New York',
            'interests' => ['coding', 'reading'],
            'settings' => ['notifications' => true],
        ], [
            'Authorization' => "Bearer $token",
        ]);

        $response->assertCreated();
        $response->assertJsonStructure([
            'id', 'user_id', 'gender', 'birth_date', 'city', 'interests', 'settings',
        ]);

        $this->assertDatabaseHas('profiles', [
            'user_id' => $user->id,
            'gender' => 'male',
            'city' => 'New York',
        ]);
    }

    public function test_profile_requires_gender(): void
    {
        $user = User::factory()->create();
        $token = $user->createToken('auth-token')->plainTextToken;

        $response = $this->postJson('/api/v1/profile', [
            'birth_date' => '1990-01-15',
            'city' => 'New York',
        ], [
            'Authorization' => "Bearer $token",
        ]);

        $response->assertUnprocessable();
        $response->assertJsonValidationErrors('gender');
    }

    public function test_profile_validates_gender_values(): void
    {
        $user = User::factory()->create();
        $token = $user->createToken('auth-token')->plainTextToken;

        $response = $this->postJson('/api/v1/profile', [
            'gender' => 'invalid',
            'birth_date' => '1990-01-15',
        ], [
            'Authorization' => "Bearer $token",
        ]);

        $response->assertUnprocessable();
        $response->assertJsonValidationErrors('gender');
    }

    public function test_profile_validates_birth_date(): void
    {
        $user = User::factory()->create();
        $token = $user->createToken('auth-token')->plainTextToken;

        $response = $this->postJson('/api/v1/profile', [
            'gender' => 'female',
            'birth_date' => 'invalid-date',
        ], [
            'Authorization' => "Bearer $token",
        ]);

        $response->assertUnprocessable();
        $response->assertJsonValidationErrors('birth_date');
    }

    public function test_authenticated_user_can_get_profile(): void
    {
        $user = User::factory()->create();
        $user->profile()->create([
            'gender' => 'male',
            'birth_date' => '1990-01-15',
            'city' => 'New York',
            'interests' => ['coding', 'reading'],
        ]);

        $token = $user->createToken('auth-token')->plainTextToken;

        $response = $this->getJson('/api/v1/profile', [
            'Authorization' => "Bearer $token",
        ]);

        $response->assertOk();
        $response->assertJsonPath('data.gender', 'male');
        $response->assertJsonPath('data.city', 'New York');
    }

    public function test_authenticated_user_can_update_profile(): void
    {
        $user = User::factory()->create();
        $user->profile()->create([
            'gender' => 'male',
            'birth_date' => '1990-01-15',
            'city' => 'New York',
        ]);

        $token = $user->createToken('auth-token')->plainTextToken;

        $response = $this->putJson('/api/v1/profile', [
            'gender' => 'female',
            'birth_date' => '1995-06-20',
            'city' => 'Los Angeles',
            'interests' => ['music', 'travel'],
        ], [
            'Authorization' => "Bearer $token",
        ]);

        $response->assertOk();
        $response->assertJsonPath('data.gender', 'female');
        $response->assertJsonPath('data.city', 'Los Angeles');

        $this->assertDatabaseHas('profiles', [
            'user_id' => $user->id,
            'gender' => 'female',
            'city' => 'Los Angeles',
        ]);
    }

    public function test_unauthenticated_user_cannot_create_profile(): void
    {
        $response = $this->postJson('/api/v1/profile', [
            'gender' => 'male',
        ]);

        $response->assertUnauthorized();
    }

    public function test_unauthenticated_user_cannot_get_profile(): void
    {
        $response = $this->getJson('/api/v1/profile');

        $response->assertUnauthorized();
    }
}
